set -e

function yes_or_no {
    while true; do
        read -p "$* [y/n]: " yn
        case $yn in
            [Yy]*) return 0  ;;
            [Nn]*) echo "" ; return  1 ;;
        esac
    done
}

function checkDependencies {
      # check if docker is installed
      if ! command -v docker &> /dev/null
      then
        echo "This installer requires 'docker'. Install it please, and then run this installer again."
        exit 1
      fi

      # check if docker-compose is installed
      if ! command -v docker-compose &> /dev/null
      then
        echo "This installer requires 'docker-compose'. Install it please, and then run this installer again."
        exit 1
      fi
}

# Create a backup of the docker volume before upgrading
function createVolumeBackup() {
  backupPath=$1
  OLD_VOLUME_NAME=$2
  echo "##############################"
  echo "Creating backup of docker volume with name $OLD_VOLUME_NAME at $backupPath"
  echo "##############################"
  # Create an archive of the volume's contents
  timestamp=$(date +%Y%m%d-%H%M%S)
  docker run --rm -v $OLD_VOLUME_NAME:/volume -v $backupPath:/backup alpine tar cf /backup/portrait-index-data-volume-$timestamp.tar /volume
}

function restoreVolumeBackup {
  backupPath=$1
  NEW_VOLUME_NAME=$2
  newInstallationPath=$3
  if docker volume ls | grep my-volume; then
    echo "Volume $NEW_VOLUME_NAME already exists. skipping restore."
  else
    docker volume create $NEW_VOLUME_NAME 
    backupFile=$(find $backupPath -name 'portrait-index-data-volume-*.tar' | tail -1)
    docker run --rm -v $NEW_VOLUME_NAME:/volume -v $backupFile:/restore/portrait-index-data-volume-latest.tar alpine /bin/sh -c "mkdir /unpack && tar -xf /restore/portrait-index-data-volume-latest.tar -C /unpack && cp -a /unpack/. /volume"
    echo "Restored backup to new docker volume"
  fi
}

function checkRequiredEnvVariables {
    newInstallationPath=$1
    requiredVars=("PROXY_PORT" "CODE_SERVER_PW")
    for var in "${requiredVars[@]}"
    do
        if grep -q $var $newInstallationPath/.env; then
            echo "$var is set - OK"
        else
            read -p "WARN $var is not set. Please provide a value: "  newVarValue
            echo "$var=$newVarValue" >> $newInstallationPath/.env
            echo "added $var=$newVarValue to .env"
        fi
    done
}

function main {

  NEW_VERSION=$(grep -E '^BACKEND_TAG=' .env | cut -d '=' -f2)

  echo ""
  echo "##############################"
  echo "Starting setup to upgrading an older version of Portrait App to version $NEW_VERSION"
  echo "Please make sure to have read upgrade instructions on https://docs.portrait.app/upgrade"
  echo "WARNING: The old version of Portrait will be stopped during the upgrade process and will NOT be restarted."
  echo "##############################"
  echo ""
  yes_or_no "continue?" || exit

  checkDependencies

  # set newInstallationPath to current location
  newInstallationPath=$(dirname -- "$0")
  echo "newInstallationPath: $newInstallationPath"

  read -p "Set the path of the old installation: "  oldInstallationPath
  oldInstallationPath=${oldInstallationPath:-''}
  if [  -n "$oldInstallationPath" ] && [ -f "$oldInstallationPath/docker-compose.yml" ]; then
    echo "old installation is valid"
  else
    echo "old installation path is invalid" >&2
    exit 1
  fi

  if [ "$oldInstallationPath" = "$newInstallationPath" ]; then
    echo "old and new installation cannot be the same" >&2
    exit 1
  fi

  OLD_VERSION=$(grep -E '^BACKEND_TAG=' $oldInstallationPath/.env | cut -d '=' -f2)

  echo "Detected old version: $OLD_VERSION"
  echo "Detected new version: $NEW_VERSION"
  echo "Please make sure the version are upgradeable according to our compatibility matrix: https://docs.portrait.app/upgrade"
  yes_or_no "continue?" || exit


  # Get the name of the docker volume that contains the index data
  read -p "Set the name of the docker volume to restore (leave empty for automatic detection): "  OLD_VOLUME_NAME
  if [ -z "$OLD_VOLUME_NAME" ]; then
    echo "Starting old installation to get the name of the docker index data volume"
    env $(cat $oldInstallationPath/.env | xargs) docker-compose -f  $oldInstallationPath/docker-compose.yml up -d index
    OLD_VOLUME_NAME=$(docker inspect $(env $(cat $oldInstallationPath/.env | xargs) docker-compose -f  $oldInstallationPath/docker-compose.yml ps -q index) | grep -A1 portrait-index-data | grep "Name" | awk '{print $2}' | tr -d ',' | sed 's/"//g')
    if [ -z "$OLD_VOLUME_NAME" ]; then
      echo "name of the docker index volume cannot be found" >&2
      exit 1
    fi
    else
      # check if the volume exists
      if ! docker volume ls | grep $OLD_VOLUME_NAME; then
        echo "volume $OLD_VOLUME_NAME does not exist" >&2
        exit 1
      fi
  fi
  echo "detected index data volume name: $OLD_VOLUME_NAME"

  echo "##############################"
  echo "Starting upgrade ..."
  echo "Shutdown old portrait installation ..."

  env $(cat $oldInstallationPath/.env | xargs) docker-compose -f $oldInstallationPath/docker-compose.yml down

  # Create a backup of the docker volume before upgrading
  createVolumeBackup $oldInstallationPath/backup $OLD_VOLUME_NAME

  # Copy Scripts
  SCRIPTING_POSTFIX="/app/config/scripting"
  echo "##############################"
  echo "Copying scripts $oldInstallationPath$SCRIPTING_POSTFIX to $newInstallationPath$SCRIPTING_POSTFIX "
  cp -r $oldInstallationPath$SCRIPTING_POSTFIX  $newInstallationPath$SCRIPTING_POSTFIX || :

  # Copy email-templates folder
  EMAIL_TEMPLATES_POSTFIX="/app/config/email-templates"
  echo "##############################"
  echo "Copying email-templates $oldInstallationPath$EMAIL_TEMPLATES_POSTFIX to $newInstallationPath$EMAIL_TEMPLATES_POSTFIX "
  cp -r $oldInstallationPath$EMAIL_TEMPLATES_POSTFIX  $newInstallationPath$EMAIL_TEMPLATES_POSTFIX || :

  # Copy Frontend favicons folder
  FRONTEND_favicons_POSTFIX="/app/config/frontend/public/favicons"
  echo "##############################"
  echo "Copying frontend favicons $oldInstallationPath$FRONTEND_favicons_POSTFIX to $newInstallationPath$FRONTEND_favicons_POSTFIX "
  cp -r $oldInstallationPath$FRONTEND_favicons_POSTFIX  $newInstallationPath$FRONTEND_favicons_POSTFIX || :

  # Copy Frontend icons folder
  FRONTEND_icons_POSTFIX="/app/config/frontend/public/icons"
  echo "##############################"
  echo "Copying frontend icons $oldInstallationPath$FRONTEND_icons_POSTFIX to $newInstallationPath$FRONTEND_icons_POSTFIX "
  cp -r $oldInstallationPath$FRONTEND_icons_POSTFIX  $newInstallationPath$FRONTEND_icons_POSTFIX || :

  echo "##############################"
  echo "FRONTEND: update .env in /app/config/frontend"
  # Set baseurl in frontend .env
  migrateFrontendVariables=("NEXT_PUBLIC_PUBLIC_URL" "NEXT_PUBLIC_API_URL" "NEXT_PUBLIC_PUBLIC_ACCESS")
  for var in "${migrateFrontendVariables[@]}"
  do
    varVal=$(grep -E "^$var=" $oldInstallationPath/app/config/frontend/.env | cut -d '=' -f2)
    sed -i "s|^$var=.*|$var=$varVal|" $newInstallationPath/app/config/frontend/.env
  done

  # Handling docker-compose
  echo "##############################"
  echo "INFO: docker-compose.yml is not migrated any manual changes will be ignored."
  yes_or_no "continue?" || exit



  # Handling .env
  {
      echo "##############################"
      echo "DOCKER: Starting migration on .env"
      # backup original .env
      cp $newInstallationPath/.env $newInstallationPath/.env.bak
      # copy .env to new installation
      docker_env_POSTFIX="/.env"
      echo "Copying .env $oldInstallationPath$docker_env_POSTFIX to $newInstallationPath$docker_env_POSTFIX "
      cp -r $oldInstallationPath$docker_env_POSTFIX  $newInstallationPath$docker_env_POSTFIX

      upgradeVersionTags=("BACKEND_TAG" "FRONTEND_TAG" "PYTHON_RUNTIME_TAG")
      for var in "${upgradeVersionTags[@]}"
      do
        if ! grep -q "^$var=" $newInstallationPath/.env; then
          echo "INSERT $var to $NEW_VERSION in .env"
          echo "$var=$NEW_VERSION" >> $newInstallationPath/.env
         else
          echo  "Set $var to $NEW_VERSION in .env"
          sed -i "s/^$var=.*/$var=""$NEW_VERSION"'/' "$newInstallationPath"/.env
        fi
      done
  }

  # Check if .env contains COMPOSE_PROJECT_NAME
  if ! grep -q '^COMPOSE_PROJECT_NAME=' $newInstallationPath/.env; then
    read -p "COMPOSE_PROJECT_NAME is missing in .env: Define a custom name for the docker project (leave blank for auto-generated) allowed characters [a-z0-9]:"  dockerProjectName
    dockerProjectName=$(echo "$dockerProjectName" |  tr '[:upper:]' '[:lower:]' | tr -dc 'a-zA-Z0-9')
    if [ -z "$dockerProjectName" ];
    then
        dockerProjectName=portrait-$(uuidgen | tr -d '-' )
        echo "Using the auto-generated name $dockerProjectName"
    fi
    echo "COMPOSE_PROJECT_NAME="$dockerProjectName >> $newInstallationPath/.env
  fi
  dockerProjectName=$(grep -E '^COMPOSE_PROJECT_NAME=' $newInstallationPath/.env | cut -d '=' -f2)

  checkRequiredEnvVariables $newInstallationPath

  echo "##############################"
  # Restore the backup of the docker volume
  NEW_VOLUME_NAME=$dockerProjectName'_portrait-index-data'
  echo "Restoring index data volume backup from $oldInstallationPath/backup to $NEW_VOLUME_NAME"
  restoreVolumeBackup $oldInstallationPath/backup $NEW_VOLUME_NAME $newInstallationPath

  # copy application.yml to new installation
  application_yml_POSTFIX="/app/config/application-prod.yml"
  echo "Copying application-prod.yml $oldInstallationPath$application_yml_POSTFIX to $newInstallationPath$application_yml_POSTFIX "
  cp -r $oldInstallationPath$application_yml_POSTFIX  $newInstallationPath$application_yml_POSTFIX
  echo "WARNING: application-prod.yml is only copied from old installation please make sure to update the config according to the new version! You can do this later but maybe the application will crash until you do this."


  echo ""
  echo "##############################"
  echo "Upgrade from $OLD_VERSION to $NEW_VERSION finished"
  echo "After you editing the application-prod.yml to your needs you can start the new version with the following command:"
  echo "docker-compose up -d"
  echo "##############################"
  echo ""

}


main